# -*- coding: utf-8 -*-

"""
Script autônomo para conversão de arquivos de texto (.txt) em ePubs otimizados.

Versão: 9.4 (Restauração Estável e Definitiva)
Data: 14 de agosto de 2025

Funcionalidades (v9.4):
- ARQUITETURA RESTAURADA: A função `extract_sentences_from_text` foi completamente
  revertida para a arquitetura estável (pré-v8.2), que separa o processamento
  de prosa e linhas especiais, eliminando a causa dos travamentos.
- TODAS AS REGRAS INTELIGENTES PRESERVADAS: O motor interno de quebra de frases
  mantém todas as regras de precisão (abreviações, diálogos, atribuições) que
  foram desenvolvidas e aprovadas.
- ESTABILIDADE MÁXIMA: Esta versão foi construída com o único objetivo de ser
  robusta, estável e livre de travamentos, seguindo as diretrizes originais.
"""

import os
import re
import chardet
from ebooklib import epub
import math

# --- PARÂMETROS DE AJUSTE (TUNING) ---
POETRY_LINE_LENGTH_THRESHOLD = 40
TARGET_SENTENCES_PER_CHAPTER = 100

# --- Módulo de Processamento de Texto (v9.4) ---

def _split_paragraph_into_sentences(paragraph):
    """
    Motor interno de quebra de frases. Contém a coleção de regras de proteção e
    reposicionamento. É chamado com blocos de texto (parágrafos lógicos).
    """
    text = re.sub(r'\s*\n\s*', ' ', paragraph).strip()
    text = re.sub(r'([\.!?;:]+)', r'\1<SPLIT>', text)
    text = re.sub(r'(\d):<SPLIT>(\d)', r'\1:\2', text)
    text = re.sub(r'((?<!\w)[A-Z][a-zA-Z]{0,2})\.<SPLIT>', r'\1.', text, flags=re.UNICODE)
    text = re.sub(r'(\d)\.<SPLIT>(\d)', r'\1.\2', text)
    text = re.sub(r'([\.!?])<SPLIT>(\s*[-—]\s*[a-z])', r'\1\2', text, flags=re.UNICODE)
    closing_delimiters_optional = r'[\)\]"\'’”]*'
    pattern_to_heal = f'([\.!?])({closing_delimiters_optional})<SPLIT>(\\s*,\\s*[a-z])'
    text = re.sub(pattern_to_heal, r'\1\2\3', text, flags=re.UNICODE)
    unambiguous_closers = r'[\)\]’”]'
    text = re.sub(f'<SPLIT>(\\s*{unambiguous_closers}+)', r'\1<SPLIT>', text)
    ambiguous_quotes = r'["\']'
    text = re.sub(f'<SPLIT>(\\s*(?<!\\s){ambiguous_quotes})', r'\1<SPLIT>', text)
    sentences = [s.strip() for s in text.split('<SPLIT>') if s.strip()]
    return sentences

def extract_sentences_from_text(text_content):
    """
    Fase 1: Extrai as frases. ARQUITETURA ESTÁVEL RESTAURADA.
    Esta lógica separa o fluxo em "prosa" (acumulada) e "linhas especiais"
    (processadas individualmente), evitando o acúmulo excessivo de memória.
    """
    sentences = []
    prose_buffer = ""
    lines = text_content.splitlines()
    for line in lines:
        line = line.strip()
        if not line:
            # Ignora linhas em branco, conforme especificação original.
            continue
        
        is_short = len(line) <= POETRY_LINE_LENGTH_THRESHOLD
        ends_with_punctuation = any(line.endswith(p) for p in ['.', '!', '?', ';', ':'])
        is_poetry_or_title = is_short and not ends_with_punctuation
        is_dialogue_start = line.startswith(('-', '—'))
        
        is_special_line = is_poetry_or_title or is_dialogue_start

        if is_special_line:
            # Uma linha especial finaliza o parágrafo de prosa anterior.
            if prose_buffer:
                sentences.extend(_split_paragraph_into_sentences(prose_buffer))
                prose_buffer = ""
            # E então a linha especial é processada individualmente.
            # Esta linha pode conter múltiplas frases e precisa ser processada.
            sentences.extend(_split_paragraph_into_sentences(line))
        else:
            # Linhas de prosa normais são acumuladas no buffer.
            if prose_buffer:
                prose_buffer += " "
            prose_buffer += line
    
    # Processa o que sobrou no buffer de prosa no final do arquivo.
    if prose_buffer:
        sentences.extend(_split_paragraph_into_sentences(prose_buffer))
    
    return sentences

def _enforce_line_start_rule(sentences):
    """
    Fase 2: Aplica a "Lei Absoluta" com "Fusão Inteligente".
    """
    if not sentences:
        return []
    merged_sentences = []
    for sentence in sentences:
        if not sentence:
            continue
        first_char = sentence[0]
        is_valid_start = False
        if first_char.isupper() or first_char.isdigit():
            is_valid_start = True
        else:
            valid_symbols = "-—([\"“‘"
            if first_char in valid_symbols:
                is_valid_start = True
        if not is_valid_start and merged_sentences:
            if any(char.isalnum() for char in sentence):
                merged_sentences[-1] += " " + sentence
            else:
                merged_sentences[-1] += sentence
        else:
            merged_sentences.append(sentence)
    return merged_sentences

# --- Módulo de Geração do ePub ---

def create_epub_from_sentences(sentences, original_filename):
    """
    Gera o ePub final com base nas frases processadas.
    """
    book_title = os.path.splitext(original_filename)[0]
    epub_filename = f"{book_title}.epub"
    book = epub.EpubBook()
    
    book.set_identifier(f"urn:uuid:{book_title}-{int(os.times().user * 1000)}")
    book.set_title(book_title)
    book.set_language("pt-br")
    book.add_author("Autor Desconhecido")

    total_sentences = len(sentences)
    if total_sentences == 0:
        return None

    total_chapters = math.ceil(total_sentences / TARGET_SENTENCES_PER_CHAPTER)
    
    chapters = []
    for i in range(total_chapters):
        chapter_num_desc = total_chapters - i
        chapter_title = f"{chapter_num_desc}"
        
        start_index = i * TARGET_SENTENCES_PER_CHAPTER
        end_index = start_index + TARGET_SENTENCES_PER_CHAPTER
        chapter_sentences = sentences[start_index:end_index]
        
        if not chapter_sentences: continue

        html_content = f"<h1>{chapter_title}</h1>"
        html_content += "".join([f"<p>{s}</p>" for s in chapter_sentences])
        
        chapter_obj = epub.EpubHtml(title=chapter_title, file_name=f'chap_{chapter_num_desc}.xhtml', lang='pt-br')
        chapter_obj.set_content(html_content)
        chapters.append(chapter_obj)
        book.add_item(chapter_obj)

    book.toc = chapters
    book.spine = ['nav'] + chapters
    book.add_item(epub.EpubNcx())
    book.add_item(epub.EpubNav())

    epub.write_epub(epub_filename, book, {})
    return epub_filename

# --- Fluxo Principal do Script ---

def main():
    """
    Orquestra todo o processo de conversão.
    """
    current_directory = os.getcwd()
    txt_files = [f for f in os.listdir(current_directory) if f.lower().endswith('.txt')]
    for txt_filename in txt_files:
        original_filepath = os.path.join(current_directory, txt_filename)
        try:
            with open(original_filepath, 'rb') as f:
                raw_data = f.read()
                detected_encoding = chardet.detect(raw_data)['encoding']
                if not detected_encoding: detected_encoding = 'utf-8'
                text_content = raw_data.decode(detected_encoding, errors='ignore')
            
            raw_sentences = extract_sentences_from_text(text_content)
            final_sentences = _enforce_line_start_rule(raw_sentences)

            if final_sentences:
                epub_filepath = create_epub_from_sentences(final_sentences, txt_filename)
                if epub_filepath:
                    os.remove(original_filepath)
        except Exception:
            continue

if __name__ == "__main__":
    main()